#region Includes
using GroundhogLearning.GameKit.Utils;
using UnityEngine;
using UnityEngine.Events;
#endregion

namespace GroundhogLearning.GameKit.Transforms
{
    [AddComponentMenu(EditorMenuConfig.CATEGORY_TRANSFORMS + "Rotate To")]
    public class RotateTo : MonoBehaviour
    {
        #region Variables

        [Header("Configuration")]
        [SerializeField, Tooltip("If true, rotation begins automatically when the scene starts.")]
        private bool _autoStart = true;

        [SerializeField, Tooltip("If true, the object will rotate back and forth between start and target rotations.")]
        private bool _loop = true;

        [SerializeField, Tooltip("Speed of rotation in degrees per second.")]
        private float _rotationSpeed = 60f;

        [SerializeField, Tooltip("Euler angles (in degrees) relative to the starting rotation.")]
        private Vector3 _targetRotationOffset = new Vector3(0f, 90f, 0f);

        [SerializeField, Tooltip("Optional delay (in seconds) before reversing direction if looping is enabled.")]
        private float _pauseTime = 0f;

        [Header("Events")]
        [SerializeField, Tooltip("Event raised when the object reaches the target rotation offset")]
        private UnityEvent OnTargetRotationReached;

        [SerializeField, Tooltip("Event raised when the object reaches the initial rotation")]
        private UnityEvent OnInitialRotationReached;

        private bool _isRotating;
        private Quaternion _startRotation;
        private Quaternion _targetRotation;
        private Quaternion _currentTargetRotation;
        private float _pauseTimer;

        #endregion


        private void Start()
        {
            _startRotation = transform.rotation;
            _targetRotation = _startRotation * Quaternion.Euler(_targetRotationOffset);
            _currentTargetRotation = _targetRotation;

            if (_autoStart)
            {
                _isRotating = true;
            }
        }
        private void Update()
        {
            if (!_isRotating) return;

            var step = _rotationSpeed * Time.deltaTime;
            transform.rotation = Quaternion.RotateTowards(transform.rotation, _currentTargetRotation, step);

            if (Quaternion.Angle(transform.rotation, _currentTargetRotation) < 0.1f)
            {
                if (_loop)
                {
                    if (_pauseTime > 0f)
                    {
                        _pauseTimer += Time.deltaTime;
                        if (_pauseTimer < _pauseTime) return;
                        _pauseTimer = 0f;
                    }

                    if (_currentTargetRotation == _targetRotation)
                    {
                        _currentTargetRotation = _startRotation;
                        OnTargetRotationReached?.Invoke();
                    }
                    else
                    {
                        _currentTargetRotation = _targetRotation;
                        OnInitialRotationReached?.Invoke();
                    }
                }
                else
                {
                    _isRotating = false;
                }
            }
        }

        public void Rotate()
        {
            _currentTargetRotation = _targetRotation;
            _isRotating = true;
        }
        public void RotateBack()
        {
            _currentTargetRotation = _startRotation;
            _isRotating = true;
        }


#if UNITY_EDITOR
        private void OnDrawGizmosSelected()
        {
            var startRotation = Application.isPlaying ? _startRotation : transform.rotation;
            var endRotation = startRotation * Quaternion.Euler(_targetRotationOffset);

            Gizmos.color = Color.yellow;
            var startDir = startRotation * Vector3.forward;
            var endDir = endRotation * Vector3.forward;

            Gizmos.DrawRay(transform.position, startDir);
            Gizmos.DrawRay(transform.position, endDir);
        }
#endif
    }
}
